/*
 * $Id: silkscreen.c,v 1.24 2003/12/31 16:24:59 prussar Exp $
 *
 * Viewer - a part of Plucker, the free off-line HTML viewer for PalmOS
 * Copyright (c) 1998-2002, Mark Ian Lillywhite and Michael Nordstrom
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 */

#include <BmpGlue.h>

#include "debug.h"
#include "prefsdata.h"
#include "util.h"
#include "mainform.h"
#include "libraryform.h"

#include "silkscreen.h"


/***********************************************************************
 *
 *      Internal Constants
 *
 ***********************************************************************/
#define TOTAL_ICONS           11
#define TOOLBAR_START_X       43
#define TOOLBAR_START_Y        3


/***********************************************************************
 *
 *      Internal Types
 *
 ***********************************************************************/
typedef struct {
    UInt16        resourceId;
    WChar         asciiCode;
    RectangleType bounds[ NUM_OF_SILKSCREENS ];
} SilkIconType;


/***********************************************************************
 *
 *      Local functions
 *
 ***********************************************************************/
static void SilkScreenIdentifyType( void ) SILKSCREEN_SECTION;
static Boolean SonyV1SilkScreenSupported( void ) SILKSCREEN_SECTION;
static Err SonyV1SilkScreenInitialize( void ) SILKSCREEN_SECTION;
static SilkScreenStatus SonyV1GetSilkScreenStatus( void ) SILKSCREEN_SECTION;
static void SonyV1SetSilkScreenStatus(
                SilkScreenStatus changeTo ) SILKSCREEN_SECTION;
static Err SonyV1SilkScreenStop( void ) SILKSCREEN_SECTION;
static Boolean SonyV2SilkScreenSupported( void ) SILKSCREEN_SECTION;
static Err SonyV2SilkScreenInitialize( void ) SILKSCREEN_SECTION;
static SilkScreenStatus SonyV2GetSilkScreenStatus( void ) SILKSCREEN_SECTION;
static void SonyV2SetSilkScreenStatus(
                SilkScreenStatus changeTo ) SILKSCREEN_SECTION;
static Err SonyV2SilkScreenStop( void ) SILKSCREEN_SECTION;
static Boolean SonyV3SilkScreenSupported( void ) SILKSCREEN_SECTION;
static Err SonyV3SilkScreenInitialize( void ) SILKSCREEN_SECTION;
static SilkScreenStatus SonyV3GetSilkScreenStatus( void ) SILKSCREEN_SECTION;
static void SonyV3SetSilkScreenStatus(
                SilkScreenStatus changeTo ) SILKSCREEN_SECTION;
static Err SonyV3SilkScreenStop( void ) SILKSCREEN_SECTION;
static Boolean HanderaSilkScreenSupported( void ) SILKSCREEN_SECTION;
static SilkScreenStatus HanderaGetSilkScreenStatus( void ) SILKSCREEN_SECTION;
static void HanderaSetSilkScreenStatus(
                SilkScreenStatus changeTo ) SILKSCREEN_SECTION;
static void HanderaResetSilkScreen( void ) SILKSCREEN_SECTION;
static void HanderaUpdateSilkVerticalOffset(
                const Char* offText ) SILKSCREEN_SECTION;
static void GetButtons( void ) SILKSCREEN_SECTION;
static void DrawButtons( void ) SILKSCREEN_SECTION;
static void SetButtons( PenBtnListType* buttonList ) SILKSCREEN_SECTION;
static void SetSilkScreenFunctions( void ) SILKSCREEN_SECTION;

static Err (*HWSpecificSilkScreenInitialize)( void );
static SilkScreenStatus (*HWSpecificGetSilkScreenStatus)( void );
static void (*HWSpecificSetSilkScreenStatus)( SilkScreenStatus changeTo );
static void (*HWSpecificResetSilkScreen)( void );
static void (*HWSpecificUpdateSilkVerticalOffset)( const Char* offText );
static Err (*HWSpecificSilkScreenStop)( void );


/***********************************************************************
 *
 *      Private variables
 *
 ***********************************************************************/

static UInt16           sonySilkScreenRefNum = NULL;
static SilkScreenStatus initialSilkStatus    = SILKSCREEN_UNKNOWN;
static SilkScreenStatus lastSetSilkStatus    = SILKSCREEN_UNKNOWN;
static MemHandle        defaultButtonsHandle[ NUM_OF_SILKSCREENS ];
static SilkScreenStatus currentSilkStatus    = NULL;
static SilkIconType     iconList[ TOTAL_ICONS ];
static Coord            lastExtentX;
static Coord            lastExtentY;

static SilkScreen silkScreenType = unknownSilkScreen;





/* set the hardware-specific functions to their usable counterparts */
void SetSilkScreenFunctions( void )
{
    switch ( silkScreenType ) {
        case sonyV1SilkScreen:
            HWSpecificSilkScreenInitialize = SonyV1SilkScreenInitialize;
            HWSpecificGetSilkScreenStatus = SonyV1GetSilkScreenStatus;
            HWSpecificSetSilkScreenStatus = SonyV1SetSilkScreenStatus;
            HWSpecificResetSilkScreen = NULL;
            HWSpecificUpdateSilkVerticalOffset = NULL;
            HWSpecificSilkScreenStop = SonyV1SilkScreenStop;
            break;

        case sonyV2SilkScreen:
            HWSpecificSilkScreenInitialize = SonyV2SilkScreenInitialize;
            HWSpecificGetSilkScreenStatus = SonyV2GetSilkScreenStatus;
            HWSpecificSetSilkScreenStatus = SonyV2SetSilkScreenStatus;
            HWSpecificResetSilkScreen = NULL;
            HWSpecificUpdateSilkVerticalOffset = NULL;
            HWSpecificSilkScreenStop = SonyV2SilkScreenStop;
            break;

        case sonyV3SilkScreen:
            HWSpecificSilkScreenInitialize = SonyV3SilkScreenInitialize;
            HWSpecificGetSilkScreenStatus = SonyV3GetSilkScreenStatus;
            HWSpecificSetSilkScreenStatus = SonyV3SetSilkScreenStatus;
            HWSpecificResetSilkScreen = NULL;
            HWSpecificUpdateSilkVerticalOffset = NULL;
            HWSpecificSilkScreenStop = SonyV3SilkScreenStop;
            break;

        case handeraSilkScreen:
            HWSpecificSilkScreenInitialize = NULL;
            HWSpecificGetSilkScreenStatus = HanderaGetSilkScreenStatus;
            HWSpecificSetSilkScreenStatus = HanderaSetSilkScreenStatus;
            HWSpecificResetSilkScreen = HanderaResetSilkScreen;
            HWSpecificUpdateSilkVerticalOffset = HanderaUpdateSilkVerticalOffset;
            HWSpecificSilkScreenStop = NULL;
            break;

        default:
            HWSpecificSilkScreenInitialize = NULL;
            HWSpecificGetSilkScreenStatus = NULL;
            HWSpecificSetSilkScreenStatus = NULL;
            HWSpecificResetSilkScreen = NULL;
            HWSpecificUpdateSilkVerticalOffset = NULL;
            HWSpecificSilkScreenStop = NULL;
            break;
    }
}



/* return what type of silkscreen support we have */
static void SilkScreenIdentifyType( void )
{
    if ( SonyV3SilkScreenSupported() )
        silkScreenType = sonyV3SilkScreen;
    else if ( SonyV2SilkScreenSupported() )
        silkScreenType = sonyV2SilkScreen;
    else if ( SonyV1SilkScreenSupported() )
        silkScreenType = sonyV1SilkScreen;
    else if ( HanderaSilkScreenSupported() )
        silkScreenType = handeraSilkScreen;
    else
        silkScreenType = noSilkScreen;
}



/* return whether or not silkscreen is currently enabled or not */
SilkScreen SilkScreenType( void )
{
    return silkScreenType;
}


/* Begin Hardware-independant (global) functions */


/* Initialize the silkscreen libraries, if supported */
Err SilkScreenInitialize( void )
{
    Err     err;
    UInt16  i;

    SilkScreenIdentifyType();
    SetSilkScreenFunctions();
    if ( silkScreenType != noSilkScreen ) {
        for ( i = 0; i < NUM_OF_SILKSCREENS; i++ )
            defaultButtonsHandle[ i ] = NULL;

        if ( HWSpecificSilkScreenInitialize != NULL )
            err = HWSpecificSilkScreenInitialize();
        else
            err = errNone;

        initialSilkStatus = GetSilkScreenStatus();
        WinGetDisplayExtent( &lastExtentX, &lastExtentX );
    }
    else {
        err = errNoSilkScreen;
    }
    return err;
}



/* Get the current silkscreen setting, if supported */
SilkScreenStatus GetSilkScreenStatus( void )
{
    if ( HWSpecificGetSilkScreenStatus != NULL )
        return HWSpecificGetSilkScreenStatus();
    return SILKSCREEN_UNKNOWN;
}



/* Set the silkscreen to a specific setting, if supported */
SilkScreenStatus SetSilkScreenStatus
    (
    SilkScreenStatus changeTo
    )
{
    SilkScreenStatus previousStatus;

    previousStatus = GetSilkScreenStatus();
    
    lastSetSilkStatus = changeTo;

    if ( HWSpecificSetSilkScreenStatus != NULL )
        HWSpecificSetSilkScreenStatus( changeTo );

    return previousStatus;
}



/* Reset the silkscreen to a state similar to how we found it, if supported */
void ResetSilkScreen( void )
{
    if ( HWSpecificResetSilkScreen != NULL )
        HWSpecificResetSilkScreen();
}



/* Update the vertical offset value on the silkscreen, if supported */
void UpdateSilkVerticalOffset
    (
    const Char* offText
    )
{
    if ( HWSpecificUpdateSilkVerticalOffset != NULL )
        HWSpecificUpdateSilkVerticalOffset( offText );
}



/* Stop/close the silkscreen libraries, if supported */
Err SilkScreenStop( void )
{
    if ( initialSilkStatus != SILKSCREEN_UNKNOWN ) {
        SetSilkScreenStatus( initialSilkStatus );
        initialSilkStatus = SILKSCREEN_UNKNOWN;
    }

    if ( HWSpecificSilkScreenStop != NULL )
        HWSpecificSilkScreenStop();

    silkScreenType = noSilkScreen;

    ResetSilkScreen();

    return errNone;
}


/* End Hardware-independant functions */

/* Begin Sony-specific SilkScreen API V1 functions */


/* identify if we support sony silkscreen V1 */
static Boolean SonyV1SilkScreenSupported( void )
{
#if SONY_SILK_API >= 0x01
    Err                err;
    SonySysFtrSysInfoP sonySysFtrSysInfoP;

    err = FtrGet ( sonySysFtrCreator,
                   sonySysFtrNumSysInfoP,
                   (UInt32 *) &sonySysFtrSysInfoP );

    /* If we're on a sony system, and have access to the SilkLib.. */
    if ( err == errNone &&
         sonySysFtrSysInfoP->libr & sonySysFtrSysInfoLibrSilk &&
         sonySysFtrSysInfoP->extn & sonySysFtrSysInfoExtnSilk ) {

        MSG( _( "Sony V1 SilkScreen supported\n" ) );
        return true;
    }
#endif
    MSG( _( "Sony V1 SilkScreen not supported\n" ) );
    return false;
}



/* sony specific silkscreen initialization procedure */
static Err SonyV1SilkScreenInitialize( void )
{
    Err err;

#if SONY_SILK_API >= 0x01
    err = SysLibFind( sonySysLibNameSilk, &sonySilkScreenRefNum );

    if ( err == sysErrLibNotFound )
        err = SysLibLoad( 'libr', sonySysFileCSilkLib, &sonySilkScreenRefNum );
    if ( err == errNone )
        err = SilkLibOpen( sonySilkScreenRefNum );
    if ( err == errNone )
        err = SilkLibEnableResize( sonySilkScreenRefNum );

    if ( err == errNone ) {
        MSG( _( "Sony V1 SilkScreen Initialized\n" ) );
    }
    else {
        err = errNoSilkScreen; /* just incase SilkLibOpen fails */
        sonySilkScreenRefNum = NULL;
        MSG( _( "Sony V1 SilkScreen NOT Initialized\n" ) );
    }
#else
    sonySilkScreenRefNum = NULL;
    err = errNoSilkScreen;
#endif
    return err;
}



/* sony specific function to check if the silkscreen is minimized or not */
static SilkScreenStatus SonyV1GetSilkScreenStatus( void )
{
#if SONY_SILK_API >= 0x01

    Coord extentX;
    Coord extentY;

    WinGetDisplayExtent( &extentX, &extentY );

    switch ( extentY ) {
        case 225:
             return SILKSCREEN_DOWN;
             break;
        case 160:
             return SILKSCREEN_UP;
             break;
        case 240:
             return SILKSCREEN_NONE;
             break;
        default:
             return SILKSCREEN_UNKNOWN;
    }
#else
    return SILKSCREEN_UNKNOWN;
#endif
}



/* sony specific function to change silkscreen status */
static void SonyV1SetSilkScreenStatus
    (
    SilkScreenStatus changeTo
    )
{
#if SONY_SILK_API >= 0x01
    switch ( changeTo ) {
        case SILKSCREEN_DOWN:
             SilkLibResizeDispWin( sonySilkScreenRefNum, silkResizeToStatus );
             break;
        case SILKSCREEN_UP:
             SilkLibResizeDispWin( sonySilkScreenRefNum, silkResizeNormal );
             break;
        case SILKSCREEN_NONE:
             SilkLibResizeDispWin( sonySilkScreenRefNum, silkResizeMax );
             break;
        case SILKSCREEN_UNKNOWN:
             break;
    }
#endif
}



/* sony specific silkscreen stopping procedure */
static Err SonyV1SilkScreenStop( void )
{
#if SONY_SILK_API >= 0x01
    if ( sonySilkScreenRefNum == NULL ||
         silkScreenType != sonyV1SilkScreen ) {
        MSG( _( "Sony V1 Silkscreen doesn't need to be stopped\n" ) );
        return errNoSilkScreen;
    }

    SilkLibDisableResize( sonySilkScreenRefNum );
    SilkLibClose( sonySilkScreenRefNum );
    sonySilkScreenRefNum = NULL;
    MSG( _( "Sony V1 Silkscreen stopped\n" ) );

    return errNone;
#else
    return errNoSilkScreen;
#endif
}


/* End Sony-specific SilkScreen API V1 functions */

/* Begin Sony-specific SilkScreen API V2 functions */


/* identify if we support sony silkscreen V2 */
static Boolean SonyV2SilkScreenSupported( void )
{
#if SONY_SILK_API >= 0x02
    Err                err;
    SonySysFtrSysInfoP sonySysFtrSysInfoP;

    err = FtrGet ( sonySysFtrCreator,
                   sonySysFtrNumSysInfoP,
                   (UInt32 *) &sonySysFtrSysInfoP );

    /* If we're on a sony system, and have access to the SilkLib.. */
    if ( err == errNone &&
         sonySysFtrSysInfoP->libr & sonySysFtrSysInfoLibrSilk &&
         sonySysFtrSysInfoP->extn & sonySysFtrSysInfoExtnSilk ) {
        UInt32 vskVersion;

        err = FtrGet( sonySysFtrCreator, sonySysFtrNumVskVersion, &vskVersion );
        if ( err == errNone )
            return true;
    }
#endif
    MSG( _( "Sony V2 SilkScreen not supported\n" ) );
    return false;
}



/* sony specific silkscreen initialization procedure */
static Err SonyV2SilkScreenInitialize( void )
{
    Err err;

#if SONY_SILK_API >= 0x02
    err = SysLibFind( sonySysLibNameSilk, &sonySilkScreenRefNum );

    if ( err == sysErrLibNotFound )
        err = SysLibLoad( 'libr', sonySysFileCSilkLib, &sonySilkScreenRefNum );
    if ( err == errNone )
        err = VskOpen( sonySilkScreenRefNum );
    if ( err == errNone )
        err = VskSetState( sonySilkScreenRefNum, vskStateEnable, 1 );

    if ( err == errNone ) { 
        /* Looks good, sony silkscreen is available */
        MSG( _( "Sony V2 SilkScreen Initialized\n" ) );
    }
    else {
        err = errNoSilkScreen; /* just incase SilkLibOpen fails */
        sonySilkScreenRefNum = NULL;
        MSG( _( "Sony V2 SilkScreen NOT Initialized\n" ) );
    }
#else
    sonySilkScreenRefNum = NULL;
    err = errNoSilkScreen;
#endif
    return err;
}



/* sony specific function to check if the silkscreen is minimized or not */
static SilkScreenStatus SonyV2GetSilkScreenStatus( void )
{
#if SONY_SILK_API >= 0x02
    Err    err;
    UInt16 state;

    err = VskGetState( sonySilkScreenRefNum, vskStateResize, &state );
    if ( state == vskResizeMax )
        return SILKSCREEN_UP;
    else if ( state == vskResizeMin )
        return SILKSCREEN_DOWN;
    else if ( state == vskResizeNone )
        return SILKSCREEN_NONE;
    else
        return SILKSCREEN_UNKNOWN;
#else
    return SILKSCREEN_UNKNOWN;
#endif
}



/* sony specific function to change silkscreen status */
static void SonyV2SetSilkScreenStatus
    (
    SilkScreenStatus changeTo
    )
{
#if SONY_SILK_API >= 0x02
    switch ( changeTo ) {
        case SILKSCREEN_DOWN:
             VskSetState( sonySilkScreenRefNum, vskStateResize, vskResizeMin );
             break;
        case SILKSCREEN_UP:
             VskSetState( sonySilkScreenRefNum, vskStateResize, vskResizeMax );
             break;
        case SILKSCREEN_NONE:
             VskSetState( sonySilkScreenRefNum, vskStateResize,
                 vskResizeNone );
             break;
        case SILKSCREEN_UNKNOWN:
             break;
    }
#endif
}



/* sony specific silkscreen stopping procedure */
static Err SonyV2SilkScreenStop( void )
{
#if SONY_SILK_API >= 0x02
    if ( sonySilkScreenRefNum == NULL ||
         silkScreenType != sonyV2SilkScreen ) {
        MSG( _( "Sony V2 Silkscreen doesn't need to be stopped\n" ) );
        return errNoSilkScreen;
    }

    SilkLibDisableResize( sonySilkScreenRefNum );
    SilkLibClose( sonySilkScreenRefNum );
    sonySilkScreenRefNum = NULL;
    MSG( _( "Sony Silkscreen stopped\n" ) );

    return errNone;
#else
    return errNoSilkScreen;
#endif
}


/* End Sony-specific SilkScreen API V2 functions */

/* Begin Sony-specific SilkScreen API V3 functions */


/* identify if we support sony silkscreen V3 */
static Boolean SonyV3SilkScreenSupported( void )
{
#if SONY_SILK_API >= 0x03
    Err                err;
    SonySysFtrSysInfoP sonySysFtrSysInfoP;

    err = FtrGet ( sonySysFtrCreator,
                   sonySysFtrNumSysInfoP,
                   (UInt32 *) &sonySysFtrSysInfoP );

    /* If we're on a sony system, and have access to the SilkLib.. */
    if ( err == errNone &&
         sonySysFtrSysInfoP->libr & sonySysFtrSysInfoLibrSilk &&
         sonySysFtrSysInfoP->extn & sonySysFtrSysInfoExtnSilk ) {
        UInt32 vskVersion;

        err = FtrGet( sonySysFtrCreator, sonySysFtrNumVskVersion, &vskVersion );
        if ( err == errNone && vskVersionNum2 < vskVersion )
            return true;
    }
#endif
    MSG( _( "Sony V3 SilkScreen not supported\n" ) );
    return false;
}



/* sony specific silkscreen initialization procedure */
static Err SonyV3SilkScreenInitialize( void )
{
    Err err;

#if SONY_SILK_API >= 0x03
    err = SysLibFind( sonySysLibNameSilk, &sonySilkScreenRefNum );

    if ( err == sysErrLibNotFound )
        err = SysLibLoad( 'libr', sonySysFileCSilkLib, &sonySilkScreenRefNum );
    if ( err == errNone )
        err = VskOpen( sonySilkScreenRefNum );
    if ( err == errNone )
        err = VskSetState( sonySilkScreenRefNum, vskStateEnable, vskResizeHorizontally );

    if ( err == errNone ) { 
        /* Looks good, sony silkscreen is available */
        MSG( _( "Sony V3 SilkScreen Initialized\n" ) );
    }
    else {
        err = errNoSilkScreen; /* just incase SilkLibOpen fails */
        sonySilkScreenRefNum = NULL;
        MSG( _( "Sony V3 SilkScreen NOT Initialized\n" ) );
    }
#else
    sonySilkScreenRefNum = NULL;
    err = errNoSilkScreen;
#endif
    return err;
}



/* sony specific function to check if the silkscreen is minimized or not */
static SilkScreenStatus SonyV3GetSilkScreenStatus( void )
{
#if SONY_SILK_API >= 0x03
    Err    err;
    UInt16 state;

    err = VskGetState( sonySilkScreenRefNum, vskStateResize, &state );
    if ( state == vskResizeMax )
        return SILKSCREEN_UP;
    else if ( state == vskResizeMin )
        return SILKSCREEN_DOWN;
    else if ( state == vskResizeNone )
        return SILKSCREEN_NONE;
    else
        return SILKSCREEN_UNKNOWN;
#else
    return SILKSCREEN_UNKNOWN;
#endif
}



/* sony specific function to change silkscreen status */
static void SonyV3SetSilkScreenStatus
    (
    SilkScreenStatus changeTo
    )
{
#if SONY_SILK_API >= 0x03
    switch ( changeTo ) {
        case SILKSCREEN_DOWN:
             VskSetState( sonySilkScreenRefNum, vskStateResize, vskResizeMin );
             break;
        case SILKSCREEN_UP:
             VskSetState( sonySilkScreenRefNum, vskStateResize, vskResizeMax );
             break;
        case SILKSCREEN_NONE:
             VskSetState( sonySilkScreenRefNum, vskStateResize,
                 vskResizeNone );
             break;
        case SILKSCREEN_UNKNOWN:
             break;
    }
#endif
}



/* sony specific silkscreen stopping procedure */
static Err SonyV3SilkScreenStop( void )
{
#if SONY_SILK_API >= 0x03
    if ( sonySilkScreenRefNum == NULL ||
         silkScreenType != sonyV3SilkScreen ) {
        MSG( _( "Sony V3 Silkscreen doesn't need to be stopped\n" ) );
        return errNoSilkScreen;
    }

    SilkLibDisableResize( sonySilkScreenRefNum );
    SilkLibClose( sonySilkScreenRefNum );
    sonySilkScreenRefNum = NULL;
    MSG( _( "Sony Silkscreen stopped\n" ) );

    return errNone;
#else
    return errNoSilkScreen;
#endif
}


/* End Sony-specific SilkScreen API V3 functions */

/* Begin Handera-specific SilkScreen API functions */


/* Identify if we support handera silkscreen */
static Boolean HanderaSilkScreenSupported( void )
{
#ifdef HAVE_HANDERA_SDK
    Err    err;
    UInt32 silkVersion;

    err = FtrGet( TRGSysFtrID, TRGSilkFtrNum, &silkVersion );

    if ( err == errNone ){
        MSG( _( "Handera SilkScreen supported\n" ) );
        return true;
    }
#endif
    MSG( _( "Handera SilkScreen not supported\n" ) );
    return false;
}



/* handera specific function to check if the silkscreen is minimized or not */
static SilkScreenStatus HanderaGetSilkScreenStatus( void )
{
#ifdef HAVE_HANDERA_SDK
    if ( SilkWindowMaximized() )
        return SILKSCREEN_UP;
    else
        return SILKSCREEN_DOWN;
#else
    return SILKSCREEN_UNKNOWN;
#endif
}



/* handera specific function to change silkscreen status */
static void HanderaSetSilkScreenStatus
    (
    SilkScreenStatus changeTo
    )
{
#ifdef HAVE_HANDERA_SDK
    if ( changeTo == SILKSCREEN_DOWN )
        SilkMinimizeWindow();
    else if ( changeTo == SILKSCREEN_UP )
        SilkMaximizeWindow();
#endif
}



/* Reset the toolbar (both minimized and maximized) to its default values */
static void HanderaResetSilkScreen( void )
{
#ifdef HAVE_HANDERA_SDK
    PenBtnListType* defaultButtons;
    UInt16          i;

    SilkRestoreDefaultTemplates();
    for ( i = 0; i < NUM_OF_SILKSCREENS; i++ ) {
        if ( defaultButtonsHandle[ i ] != NULL ) {
            defaultButtons = MemHandleLock( defaultButtonsHandle[ i ] );
            SilkSetButtonList( defaultButtons, (Boolean) i );
            SafeMemPtrFree( defaultButtons );
            defaultButtonsHandle[ i ] = NULL;
        }
    }
#endif
}



static void HanderaUpdateSilkVerticalOffset
    (
    const Char* offText
    )
{
#ifdef HAVE_HANDERA_SDK
    WinHandle silkWindow, origWindow;
    FontID    origFont;

    silkWindow = SilkGetWindow();
    origWindow = WinSetDrawWindow( silkWindow );
    origFont   = FntSetFont( stdFont );

    WinEraseRectangle( &( iconList[ 1 ].bounds[ currentSilkStatus ] ), 0 );
    WinDrawChars( offText, StrLen( offText ),
        iconList[ 1 ].bounds[ currentSilkStatus ].topLeft.x,
        iconList[ 1 ].bounds[ currentSilkStatus ].topLeft.y );

    FntSetFont( origFont );
    WinSetDrawWindow( origWindow );
#endif
}


/* End Handera-specific SilkScreen API functions */


/* function to populate the iconList with valid data */
static void GetButtons( void )
{
    iconList[ 0 ].resourceId = bmpDbase;
    iconList[ 0 ].asciiCode  = vchrPluckerDbase;

    iconList[ 1 ].resourceId = 0;
    iconList[ 1 ].asciiCode  = vchrPluckerOffset;

    iconList[ 2 ].resourceId = bmpBookmark;
    iconList[ 2 ].asciiCode  = vchrPluckerBookmark;

    iconList[ 3 ].resourceId = bmpFind;
    iconList[ 3 ].asciiCode  = vchrPluckerFind;

    iconList[ 4 ].resourceId = bmpAgain;
    iconList[ 4 ].asciiCode  = vchrPluckerAgain;

    iconList[ 5 ].resourceId = bmpAutoscrollDecr;
    iconList[ 5 ].asciiCode  = vchrPluckerAutoscrollDecr;

    iconList[ 6 ].resourceId = bmpAutoscrollStop;
    iconList[ 6 ].asciiCode  = vchrPluckerAutoscrollStop;

    iconList[ 7 ].resourceId = bmpAutoscrollIncr;
    iconList[ 7 ].asciiCode  = vchrPluckerAutoscrollIncr;

    iconList[ 8 ].resourceId = bmpLeft;
    iconList[ 8 ].asciiCode  = vchrPluckerLeft;

    iconList[ 9 ].resourceId = bmpHome;
    iconList[ 9 ].asciiCode  = vchrPluckerHome;

    iconList[ 10 ].resourceId = bmpRight;
    iconList[ 10 ].asciiCode  = vchrPluckerRight;

/* FIXME: Maybe used later, still have to figure out how to :) */
#if 0
    iconList[ 11 ].resourceId = bmpAutoscrollStart;
    iconList[ 11 ].asciiCode  = vchrPluckerAutoscrollStart;

    iconList[ 12 ].resourceId = bmpWait;
    iconList[ 12 ].asciiCode  = vchrPluckerWait;
#endif
}



/* Set the silkscreen to contain the toolbar buttons */
void SetSilkScreen( void )
{
    UInt16          defaultSize;
    PenBtnListType* defaultButtons;
    UInt16          numNewButtons;
    UInt16          newSize;
    PenBtnListType* newButtons;

    /* currently only handeraSilkScreen supported */
    if ( silkScreenType != handeraSilkScreen )
        return;

    currentSilkStatus = GetSilkScreenStatus();

    /* if we already have defaultButtonsHandle stored in memory, there
       is no point in doing everything again, since its already done */
    if ( defaultButtonsHandle[ currentSilkStatus ] != NULL )
        return;

#ifdef HAVE_HANDERA_SDK
    if ( silkScreenType == handeraSilkScreen )
        defaultSize = SilkGetButtonListSize( (Boolean) currentSilkStatus );
    else
#endif
/* setting defaultSize = 0 is bogus, because we won't ever get to this
   point unless HAVE_HANDERA_SDK is defined, and silkScreenType =
   handeraSilkScreen. Later on we'll figure out how to actually do this
   properly, for now, its in here to avoid compilation problems */
#if SONY_SILK_API >= 0x01
    if ( silkScreenType == sonyV1SilkScreen ||
         silkScreenType == sonyV2SilkScreen ||
         silkScreenType == sonyV3SilkScreen )
        defaultSize = NULL;
    else
#endif
    defaultSize = NULL;

    defaultButtonsHandle[ currentSilkStatus ] = MemHandleNew( defaultSize );
    defaultButtons = MemHandleLock( defaultButtonsHandle[ currentSilkStatus ] );
#ifdef HAVE_HANDERA_SDK
    SilkGetButtonList( defaultButtons, (Boolean) currentSilkStatus );
#endif

    numNewButtons = defaultButtons->numButtons + TOTAL_ICONS;
    newSize = sizeof( PenBtnListType ) +
        ( sizeof( PenBtnInfoType ) * ( numNewButtons - 1 ) );
    newButtons = SafeMemPtrNew( newSize );
    MemMove( newButtons, defaultButtons, defaultSize );

    GetButtons();
    DrawButtons();
    SetButtons( newButtons );
#ifdef HAVE_HANDERA_SDK
    SilkSetButtonList( newButtons, (Boolean) currentSilkStatus );
#endif

    MemPtrUnlock( defaultButtons );
    SafeMemPtrFree( newButtons );

}



/* Draw the actual buttons onto blank bitmaps, and set them as valid
   templates on the silkscreen */
static void DrawButtons( void )
{
    Err       err;
    BitmapPtr silkBitmap;
    BitmapPtr silkBitmapInv;
    WinHandle silkWindow;
    WinHandle silkWindowInv;
    WinHandle origWindow;
    UInt16    i;
    Coord     currentX;
    Coord     currentY;
    Coord     silkX;
    Coord     silkY;

    origWindow = WinGetDrawWindow();

    if ( currentSilkStatus == SILKSCREEN_UP ) {
#ifdef HAVE_HANDERA_SDK
        if ( silkScreenType == handeraSilkScreen ) {
            SilkGetTemplateBitmaps( &silkBitmap, &silkBitmapInv, NULL, NULL );
        }
        else
#endif
        {
            silkBitmap = NULL;
            silkBitmapInv = NULL;
        }
    }
    else if ( currentSilkStatus == SILKSCREEN_DOWN ) {
#ifdef HAVE_HANDERA_SDK
        if ( silkScreenType == handeraSilkScreen ) {
            SilkGetTemplateBitmaps( NULL, NULL, &silkBitmap, &silkBitmapInv );
        }
        else
#endif
        {
            silkBitmap = NULL;
            silkBitmapInv = NULL;
        }
    }
    else {
        return;
    }
    BmpGlueGetDimensions( silkBitmap, &silkX, &silkY, NULL );
    silkWindow = WinCreateOffscreenWindow( silkX, silkY, screenFormat, &err );
    silkWindowInv = WinCreateOffscreenWindow( silkX, silkY, screenFormat, &err);
    WinSetDrawWindow( silkWindow );
    WinDrawBitmap( silkBitmap, 0, 0 );
    WinSetDrawWindow( silkWindowInv );
    WinDrawBitmap( silkBitmapInv, 0, 0 );

    /* We need to move down the existing silkscreen to make room for our
       own toolbar's buttons */
    if ( currentSilkStatus == SILKSCREEN_UP ) {
        RectangleType area;
        UInt16        moveY;

        area.topLeft.x = 40;
        area.topLeft.y = 1;
        area.extent.x = 200;
        area.extent.y = 18;
        moveY         = 14;

        WinCopyRectangle( silkWindow, silkWindow, &area,
            area.topLeft.x, moveY, winPaint );
        WinCopyRectangle( silkWindowInv, silkWindowInv, &area,
            area.topLeft.x, moveY, winPaint );
        area.extent.y = moveY;
        WinSetDrawWindow( silkWindow );
        WinEraseRectangle( &area, 0 );
        WinSetDrawWindow( silkWindowInv );
        WinEraseRectangle( &area, 0 );
    }
    currentX = TOOLBAR_START_X;
    currentY = TOOLBAR_START_Y;

    for ( i = 0; i < TOTAL_ICONS; i++ ) {
        MemHandle bitmapH;
        BitmapPtr bitmap;
        Coord     width;
        Coord     height;

        if ( iconList[ i ].resourceId == 0 ) {
            /* This is just a placeholder for our '0' resourced offset image */
            width = 22;
            height = 13;
        }
        else {
            bitmapH = DmGetResource( bitmapRsc, iconList[ i ].resourceId );
            bitmap = MemHandleLock( bitmapH );
            BmpGlueGetDimensions( bitmap, &width, &height, NULL );

            WinSetDrawWindow( silkWindow );
            WinDrawBitmap( bitmap, currentX, currentY );
            WinSetDrawWindow( silkWindowInv );
            WinDrawBitmap( bitmap, currentX, currentY );

            MemPtrUnlock( bitmap );
            DmReleaseResource( bitmapH );
        }

        iconList[ i ].bounds[ currentSilkStatus ].topLeft.x = currentX;
        iconList[ i ].bounds[ currentSilkStatus ].topLeft.y = currentY;
        iconList[ i ].bounds[ currentSilkStatus ].extent.x = width;
        iconList[ i ].bounds[ currentSilkStatus ].extent.y = height;
        WinInvertRectangle( &( iconList[ i ].bounds[ currentSilkStatus ]), 0 );

        /* Because some icons are meant to appear right beside each other,
           they're defined here up top. Everything else is spaced so
           it fits nicely */
        switch ( iconList[ i ].resourceId ) {
            case bmpFind:
            case bmpAutoscrollDecr:
            case bmpAutoscrollStop:
            case bmpLeft:
            case bmpHome:
                currentX += width;
                break;

            default:
                currentX += width;
                if ( currentSilkStatus == SILKSCREEN_UP )
                    currentX += 3;
                else if ( currentSilkStatus == SILKSCREEN_DOWN )
                    currentX += 7;
                break;
        }
    }
    WinSetDrawWindow( origWindow );

    silkBitmap = WinGetBitmap( silkWindow );
    silkBitmapInv = WinGetBitmap( silkWindowInv );

#ifdef HAVE_HANDERA_SDK
    if ( currentSilkStatus == SILKSCREEN_UP )
        SilkSetTemplateBitmaps( silkBitmap, silkBitmapInv, NULL, NULL );
    else
        SilkSetTemplateBitmaps( NULL, NULL, silkBitmap, silkBitmapInv );
#endif

    WinDeleteWindow( silkWindow, false );
    WinDeleteWindow( silkWindowInv, false );
}



/* Set the buttons on the silkscreen so that our new images are valid */
static void SetButtons
    (
    PenBtnListType* buttonList /* pointer to PenBtnListType structure */
    )
{
    UInt16 index, i;

    index = buttonList->numButtons;

    if ( currentSilkStatus == SILKSCREEN_UP ) {
        RectangleType alpha, numeric;

        /* Set the numeric input area to null so our buttons are valid */
#ifdef HAVE_HANDERA_SDK
        SilkGetAreas( &alpha, &numeric );
#endif
        alpha.topLeft.y += 14;
        alpha.extent.y -= 14;
        numeric.topLeft.y += 14;
        numeric.extent.y -= 14;
#ifdef HAVE_HANDERA_SDK
        SilkSetAreas( &alpha, &numeric );
#endif
    }

    /* Because the some virtual silkscreen buttons will overlap the area
       that we want, we need to move the pen bounds around to the actual
       icon locations.

       FIXME: there should be a better way than just guessing 2, 6, 7 or 8
       as positions we want to modify in the array */
    if ( currentSilkStatus == SILKSCREEN_UP ) {
        buttonList->buttons[ 6 ].boundsR.topLeft.y += 14;
        buttonList->buttons[ 7 ].boundsR.topLeft.y += 14;
        buttonList->buttons[ 8 ].boundsR.topLeft.y += 14;
    }
    else if ( currentSilkStatus == SILKSCREEN_DOWN ) {
        buttonList->buttons[ 2 ].boundsR.topLeft.x = 220;
        buttonList->buttons[ 2 ].boundsR.extent.x = 20;
    }

    for ( i = 0; i < TOTAL_ICONS; i++ ) {
        RectangleType boundsR;

        boundsR = iconList[ i ].bounds[ currentSilkStatus ];
        if ( currentSilkStatus == SILKSCREEN_UP )
            boundsR.topLeft.y = 3;
        else if ( currentSilkStatus == SILKSCREEN_DOWN )
            boundsR.topLeft.y = 66;
        buttonList->buttons[ index + i ].boundsR = boundsR;
        buttonList->buttons[ index + i ].asciiCode = iconList[ i ].asciiCode;
        buttonList->buttons[ index + i ].keyCode = 0;
        buttonList->buttons[ index + i ].modifiers = 0;
    }
    buttonList->numButtons += TOTAL_ICONS;
}



/* Called only by a notification event that the screen in general just changed.
   This function is designed to answer the question "Did the silkscreen REALLY
   just change, or am I just imaging things?" */
void SilkVerifyChange( void )
{
    Coord newExtentX;
    Coord newExtentY;

    WinGetDisplayExtent( &newExtentX, &newExtentY );

    /* This function may be called when the depth changes, or other such
       petty events. Since we're only interested in the silkscreen, the
       tell-tale sign that it has changed is that WinGetDisplayExtent() will
       produce a different value then the last time we called it */
    if ( newExtentX == lastExtentX && newExtentY == lastExtentY )
        /* False alarm */
        return;
    if ( lastSetSilkStatus == GetSilkScreenStatus() )
        return;
    lastSetSilkStatus = GetSilkScreenStatus();
#if SONY_SILK_API >= 0x01
    /* To avoid handera accidently hitting this, we only want to use this
       for sony devices, Handera devices are to call SilkHandleChange() only
       if the event displayExtentChangedEvent is received (see viewer.c) */
    if ( silkScreenType == sonyV1SilkScreen ||
         silkScreenType == sonyV2SilkScreen ||
         silkScreenType == sonyV3SilkScreen ) {
        SilkHandleChange();
        MSG( _( "Switching sonySilkIsMinimized\n" ) );
    }
#endif

    lastExtentX = newExtentX;
    lastExtentY = newExtentY;
}



/* Find out what form the user was just on, and translate it into its
   silk-up or silk-down counterpart */
void SilkHandleChange( void )
{
    UInt16  formId;
    Boolean inactive;
    
    inactive = ( WinGetActiveWindow() !=
                 FrmGetWindowHandle( FrmGetFormPtr( Prefs()->lastForm ) ) );

    switch ( Prefs()->lastForm ) {
        case frmMainTop:
        case frmMainTopHandera:
        case frmMainBottom:
        case frmMainBottomHandera:
        case frmMainBottomHanderaLow:
        case frmMainBottomSonyLow:
        case frmMainBottomSonyWide:
        case frmMainNone:
        case frmMainNoneHandera:
            if ( inactive ) {
                /* Set the main form to update as soon as we get back to it. */
                SetMainFormUpdate();
                return;
            }
            formId = GetMainFormId();
            break;

        case frmLibrary:
        case frmLibraryHandera:
        case frmLibraryHanderaLow:
        case frmLibrarySonyWide:
            if ( inactive ) {
                /* Set the library form to update as soon as we get back to it. */
                SetLibraryFormUpdate();
                return;
            }
            formId = GetValidForm( frmLibrary );
            break;

        default:
            /* If the form is not supported, just exit gracefully */
            return;
    }

    FrmGotoForm( formId );
}



/* Disable resizing for Sony silkscreen */
/* Does nothing on non-Sony devices */
void SonySilkDisableResize( void )
{
#if SONY_SILK_API >= 0x01
   if ( silkScreenType == sonyV1SilkScreen )
       SilkLibDisableResize( sonySilkScreenRefNum );
#endif
#if SONY_SILK_API >= 0x02
   if ( silkScreenType == sonyV2SilkScreen || silkScreenType == sonyV3SilkScreen )
      VskSetState( sonySilkScreenRefNum, vskStateEnable, 0 );
#endif
}



/* Enable resizing for Sony silkscreen */
/* Does nothing on non-Sony devices */
void SonySilkEnableResize( void )
{
#if SONY_SILK_API >= 0x01
   if ( silkScreenType == sonyV1SilkScreen )
       SilkLibEnableResize( sonySilkScreenRefNum );
#endif
#if SONY_SILK_API >= 0x02
   if ( silkScreenType == sonyV2SilkScreen )
#if SONY_SILK_API >= 0x03   
      VskSetState( sonySilkScreenRefNum, vskStateEnable, vskResizeVertically );
#else
      VskSetState( sonySilkScreenRefNum, vskStateEnable, 1 );
#endif
#endif
#if SONY_SILK_API >= 0x03
   if ( silkScreenType == sonyV3SilkScreen )
      VskSetState( sonySilkScreenRefNum, vskStateEnable, vskResizeHorizontally );
#endif
}
